/*
 * This program converts an X pixmap to a windows bitmap. It is only a
 * quick hack and it will handle up to 256 colors (you are welcome to
 * remove this limitation or add any other improvements if you want).
 * It should work fine on most 16 and 32-bit systems under most O/S
 * (I don't know what will happen in 9-bit machines). It will work
 * properly on big and little endian CPUs (Intel and Motorola), but
 * not on some weird addressing machines.
 *
 * Written by Hippocrates Sendoukas, May 1993, Los Angeles
 */
#include <stdio.h>

typedef unsigned char	uns8;
typedef unsigned short	uns16;
typedef unsigned long	uns32;

typedef struct
  {
	uns16	bfType;			/* Must be "BM" */
	uns32	bfSize;			/* Size in 32-bit words of file */
	uns16	bfReserved1;		/* Must be zero */
	uns16	bfReserved2;		/* Must be zero */
	uns32	bfOffBits;		/* Offset to actual bits */
  } BITMAPFILEHEADER;

typedef struct
  {
	uns32	biSize;			/* Number of bytes in this structure */
	uns32	biWidth;		/* Width of bitmap in pels */
	uns32	biHeight;		/* Height of bitmap in pels */
	uns16	biPlanes;		/* Must be 1 */
	uns16	biBitCount;		/* Bits per pixel */
	uns32	biCompression;		/* We will put zero: no compression */
	uns32	biSizeImage;		/* Size of image in bytes */
	uns32	biXPelsPerMeter;
	uns32	biYPelsPerMeter;
	uns32	biClrUsed;		/* Number of colors used */
	uns32	biClrImportant;		/* Number of important colors */
  } BITMAPINFOHEADER;

typedef struct
  {
	uns8	rgbBlue;
	uns8	rgbGreen;
	uns8	rgbRed;
	uns8	rgbReserved;
  } RGBQUAD;


#if defined(MSDOS) || defined(__MSDOS__)
#  include	<string.h>
#  include	<stdlib.h>
#  include	<io.h>
#  define	WB_MODE	"wb"
   int		transf(FILE *,FILE *);
   int		error(char *);
   unsigned	hextodec(unsigned,unsigned);
   void		freemat(char **);
   void		invert(uns8 *,int);
   void		fix_fileheader(BITMAPFILEHEADER *);
   void		fix_infoheader(BITMAPINFOHEADER *);
   int		smallendian(void);
#else
#  define	WB_MODE	"w"
   char		*calloc();
   void		free();
   char		*strchr();
#endif

#define	invert16(x)	invert((uns8 *)(x),sizeof(uns16))
#define invert32(x)	invert((uns8 *)(x),sizeof(uns32))


void
invert(p,n)
  uns8 *p;
  int n;
{
  int	i,j;
  uns8	c;

  for (i=0,j=n-1; i<j; i++,j--)
    {
      c		= p[i];
      p[i]	= p[j];
      p[j]	= c;
    }
}


int
smallendian()
{
  char temp[sizeof(int)];

  * (int *)temp = 1;
  return (temp[0]==1);
}


void
fix_fileheader(fh)
  BITMAPFILEHEADER *fh;
{
  if (smallendian()) return;
  invert32(&(fh->bfSize));
  invert32(&(fh->bfOffBits));
}


void
fix_infoheader(ih)
  BITMAPINFOHEADER *ih;
{
  if (smallendian()) return;
  invert32(&(ih->biSize));
  invert32(&(ih->biWidth));
  invert32(&(ih->biHeight));
  invert32(&(ih->biSizeImage));
  invert32(&(ih->biCompression));
  invert32(&(ih->biXPelsPerMeter));
  invert32(&(ih->biYPelsPerMeter));
  invert32(&(ih->biClrImportant));
  invert32(&(ih->biClrImportant));
  invert16(&(ih->biPlanes));
  invert16(&(ih->biBitCount));
}


unsigned
hextodec(h,l)
  unsigned h,l;
{
  char	*p;
  static char *s1 = "0123456789ABCDEF";
  static char *s2 = "0123456789abcdef";

  if ( (p=strchr(s1,l)) != NULL )	l = (unsigned) (p-s1);
  else if ( (p=strchr(s2,l)) != NULL )	l = (unsigned) (p-s2);
  else					l = 0;
  if ( (p=strchr(s1,h)) != NULL )	h = (unsigned) (p-s1);
  else if ( (p=strchr(s2,h)) != NULL )	h = (unsigned) (p-s2);
  else					h = 0;
  return (h << 4) | l;
}


int
error(s)
  char *s;
{
  fprintf(stderr,"xpm2bmp: %s\n",s);
  return 0;
}


void
freemat(p)
  char **p;
{
  unsigned i;

  for (i=0; p[i]; i++)	free(p[i]);
  free(p);
}


int
transf(fin,fout)
  FILE *fin,*fout;
{
  RGBQUAD		*pal;
  uns32			imagesize;
  unsigned		nx,nx2,ny,palent,nc,i,j,code_ind[256];
  char			**buf,temp[80];
  BITMAPFILEHEADER	fh;
  BITMAPINFOHEADER	ih;

  if (!fgets(temp,12,fin) || strcmp(temp,"! XPM2  \n"))
    return error("Cannot recognize the input file format");

  if (!fgets(temp,sizeof(temp)-1,fin) ||
      sscanf(temp,"%u %u %u %u",&nx,&ny,&palent,&nc)!=4)
    return error("Cannot read the dimensions of the pixmap");

  if (palent<2 || palent>256 || nc!=1)
    return error("I don't know how to handle this color format");

  for (i=0; i<sizeof(code_ind)/sizeof(code_ind[0]); i++)
    code_ind[i] = 0;

  if ( (pal=(RGBQUAD *)calloc(palent,sizeof(RGBQUAD))) == NULL )
    return error("Out of memory");

  if ( (buf=(char **)calloc(ny+1,sizeof(char *))) == NULL )
    {
      free(pal);
      return error("Out of memory");
    }

  nx2 = (nx+3) / 4;		/* 32-bit alignment is required for DIBs */
  nx2 *= 4;

  for (i=0; i<ny; i++)
    if ( (buf[i]=(char *)calloc(nx2,1)) == NULL )
      {
        freemat(buf);
        free(pal);
        return error("Out of memory");
      }

  for (i=0; i<palent; i++)
    {
      if (!fgets(temp,sizeof(temp)-1,fin) || strlen(temp)!=18 ||
          temp[1]!=' ' || temp[2]!='c' || temp[3]!=' ' || temp[4]!='#')
        {
          freemat(buf);
          free(pal);
          return error("Invalid color entry");
        }
      code_ind[(uns8)temp[0]] = i;
      pal[i].rgbRed	= hextodec(temp[7],temp[8]);
      pal[i].rgbGreen	= hextodec(temp[11],temp[12]);
      pal[i].rgbBlue	= hextodec(temp[15],temp[16]);
    }

  imagesize = (uns32) nx2 * (uns32) ny;

  strcpy((char *) &(fh.bfType),"BM");
  fh.bfReserved1	= fh.bfReserved2 = 0;
  fh.bfOffBits		= sizeof(BITMAPFILEHEADER)+sizeof(BITMAPINFOHEADER)+
				palent*sizeof(RGBQUAD);
  fh.bfSize		= imagesize + fh.bfOffBits;
  fix_fileheader(&fh);
  fwrite(&fh,sizeof(fh),1,fout);

  ih.biSize		= sizeof(ih);
  ih.biWidth		= nx;
  ih.biHeight		= ny;
  ih.biPlanes		= 1;
  ih.biBitCount		= 8;
  ih.biCompression	= 0;
  ih.biSizeImage	= imagesize;
  ih.biXPelsPerMeter	= 0;
  ih.biYPelsPerMeter	= 0;
  ih.biClrUsed		= palent;
  ih.biClrImportant	= 0;
  fix_infoheader(&ih);
  fwrite(&ih,sizeof(ih),1,fout);

  fwrite(pal,palent,sizeof(RGBQUAD),fout);

  for (i=0; i<ny; i++)
    {
      for (j=0; j<nx; j++)
        buf[i][j] = code_ind[(uns8)(getc(fin))];
      if ( (j=getc(fin)) != '\n' && ((int)j)!=EOF )
        ungetc(j,fin);
    }

  /*
   * Write out the image with reversed rows
   */
  for (j=ny; j>0; j--) fwrite(buf[j-1],nx2,1,fout);

  freemat(buf);
  free(pal);
  return 1;
}


int
main(ac,av)
  int	ac;
  char	*av[];
{
  FILE	*fin,*fout;
  int	i;

  if (ac!=3)
    {
      fprintf(stderr,"Usage: xpm2bmp <infile> <outfile>\n");
      return 1;
    }
  if ( (fin=fopen(av[1],"r")) == NULL )
    {
      fprintf(stderr,"xpm2bmp: ");
      perror(av[1]);
      return 1;
    }
  if ( (fout=fopen(av[2],WB_MODE)) == NULL )
    {
      fprintf(stderr,"xpm2bmp: ");
      perror(av[2]);
      return 1;
    }
  i = !transf(fin,fout);
  fclose(fout);
  fclose(fin);
  if (i) unlink(av[2]);
  return i;
}

