/*
 * SOURCE:  dos.c
 * PROJECT: EasyTeX
 *
 * PURPOSE: low level dos routines
 *
 * UPDATES: 08/15/1991 - major rewrite
 *          (further update notes in main.c)
 *
 * (c)M.Schollmeyer
 */
#include <dos.h>
#include <string.h>

#include "macros.h"
#include "dos.h"
                   /* static variables */
void (__cdecl __interrupt __far *oldint24)();

int int_pending = 0;
                   /*  incremented for each call of an interrupt by
                       IntPreamble() and decremented by IntPostamble()
                       tested by some functions wich cannot be called
                       during interrupt processing
                     */
int errnum = 0;

void IntPreamble( void ) {

    ++int_pending;
}

void IntPostamble( void ) {

    --int_pending;
}



void SetHandler( void (__cdecl __interrupt __far *handler)() ) {

    oldint24 = _dos_getvect( 0x24 );
    _dos_setvect( 0x24, handler );
}


void UnSetHandler( void ) {

    _dos_setvect( 0x24, oldint24 );
}


/*
;   Get Clock Counter
;   Interrupt 1ah Service 0
*/
unsigned long  GetClock( void ) {

    _asm {
        mov     ah, 00h
        int     1ah

        mov     ax, dx
        mov     dx, cx
    }
}



/*
; Get Real Time Clock
; Interrupt 1ah Service 2
*/
int GetRTC( struct RealTimeClock *rtc ) {

    if( int_pending > 0 )
        return 0;

    _asm {
        mov     ah, 2ch
        int     21h
        push    ds
        lds     bx, rtc
        mov     [bx]rtc.Hours, ch
        mov     [bx].Minutes, cl
        mov     [bx].Seconds, dh
        mov     [bx].Hundrets, dl

        mov     ah, 2ah
        int     21h

        mov     [bx]rtc.Day, dl
        mov     [bx].Month, dh
        mov     [bx].Year, cx
        mov     [bx].DayOfWeek, al
        pop     ds
    }
    return 1;
}

/*
    Get/Set Break Flag

 */
int BreakFlag( int newflag ) {

    int oldflag;

    /* get old flag */
    _asm {
        mov     ax, 3300h
        int     21h
        xor     dh, dh
        mov     oldflag, dx
    }

    if( newflag != -1 ) {
        _asm {
            mov     ax, 3301h
            mov     dx, newflag
            int     21h
        }
    }

    return oldflag;
}

/*
;   Terminate Process with Return Code
;   Interrupt 21h Service 4Ch
*/
void  ExitProc( int code ) {

    _asm {
        mov     ax, code
        mov     ah, 4ch
        int     21h
    }
}


/*
;   Allocate Segment
;   Interrupt 21h Service 48h
*/
unsigned int  AllocSeg( unsigned int numpars ) {

    if( int_pending > 0 )
        return 0;

    errnum = 0;

    _asm {
        mov     bx, numpars
        mov     ah, 48h
        int     21h
        jnc     noerr
        mov     errnum, ax
        xor     ax, ax
noerr:
        nop
    }
}

/*
;   Free Memory
;   Interrupt 21h Service 49h
*/
int FreeMem( unsigned int segment ) {

    if( int_pending )
        return 0;

    errnum = 0;

    _asm {
        mov     es, segment
        mov     ah, 49h
        int     21h
        jnc     noerr
        mov     errnum, ax
noerr:
        xor     ax, ax
    }
}


void Sound( int frequ, int dura ) {

    static void _wait( unsigned int time );

    _asm {
        xor     ax, ax
        push    ax
        call    FAR PTR _wait       ; initialise wait routine
        pop     ax                  ; initialise port
        mov     al, 182
        out     43h, al
        mov     bx, frequ
        cmp     bx, 0013h
        jl      done
        mov     dx, 0012h           ; remainder = 1190000 / freq
        mov     ax, 2870h
        div     bx
        out     42h, al             ; send it to port
        mov     al, ah
        out     42h, al
        in      al, 61h             ; read old value
        push    ax                  ; save it on stack
        or      al, 03
        out     61h, al             ; switch on sound
        mov     ax, dura
        push    ax
        call    FAR PTR _wait       ; wait duration
        pop     ax                  ; recall old port value
        pop     ax
        out     61h, al             ; put it back on port
done:
        nop
    }
}


static void _wait( unsigned int time ) {

    _asm {
        push    es
        sti
        mov     ax, 0040h
        mov     es, ax
        mov     dx, time
        mov     bx, word ptr es:[6ch]
$1:
        cmp     bx, word ptr es:[6ch]
        jz      $1
$2:
        mov     ax, word ptr es:[6ch]
        sub     ax, bx
        cmp     ax, dx
        jl      $2
        pop     es
    }
}

/*
; ********************************************************************
; *
; *     HARD DISK DRIVE/FLOPPY DISKETTE SERVICES
; *
; *
; ********************************************************************
*/

/*
;   Select Disk
;   Interrupt 21h Service Eh
*/
int  SelectDisk( int drive ) {

    errnum = 0;

    _asm {
        mov     ah, 0Eh
        mov     dx, drive
        int     21h

        xor     ah, ah
    }
    if( drive != GetCurrDisk() ) {
        errnum = 0x0f;
        return FALSE;
    }
    return TRUE;
}

/*
;   Get Number of Drives
;   Interrupt 21h Service Eh
*/
int  GetNumDrives( void ) {

    int def = GetCurrDisk();

    _asm {
        mov     ah, 0Eh
        mov     dx, def
        int     21h

        xor     ah, ah
    }
}



/*
;   Get Current Disk
;   Interrupt 21h Service 19h
*/
int  GetCurrDisk( void ) {

    _asm {
        mov     ah, 19h
        int     21h
        xor     ah, ah
    }
}


/*
;   Get Drive Data
;   Interrupt 21h Service 1Ch
*/
struct DriveData *GetDriveData( unsigned char drive ) {

    struct DriveData dd;
    _asm {
        push    ds
        mov     ah, 1ch
        mov     dl, drive
        int     21h

        push    bx
        mov     bx, offset dd
        mov     [bx]dd.SectorsPerCluster, al
        mov     [bx].SectorSize, cx
        mov     [bx].NumClusters, dx
        pop     bx
        mov     al, byte ptr [bx]
        pop     ds
        mov     [bx].MediaID, al
    }
}

/*
;   Get DTA Address
;   Interrupt 21h Service 2Fh
*/
struct DiskTransferArea  _far *GetDTAAddress( void ) {

    _asm {
        mov     ah, 2fh
        int     21h

        mov     ax, bx
        mov     dx, es
    }
}

/*
;   Set Current Directory
;   Interrupt 21h Service 3Bh
*/
unsigned int  SetCurrDir( char _far *dir ) {

    int drive = (int)_2upper(dir[0]) - (int)'A';
    int dirlen = _fstrlen( dir );

    if( dirlen > 1 && dir[dirlen-1] == '\\' && !(dirlen==3 && dir[1]==':' ) ) {
        // path terminated by '\' && path != "\" not allowed by DOS
        dir[_fstrlen(dir)-1] = '\0';
    }
    if( dir[1] == ':' ) {
        SelectDisk( drive );
        if( drive != GetCurrDisk() )
            return 0;
        /* don't try to set the current directory to a null string,
           if there is no string left after setting the drive, return ok */
        dir += 2;
        if( *dir == '\0' )
            return 1;
    }

    _asm {
        push    ds
        mov     ah, 3bh
        lds     dx, dir
        int     21h
        jnc     noc
        xor     ax, ax
        jmp     done
noc:
        mov     ax, 1
done:
        pop     ds
    }
}


/*
;   Create File
;   Interrupt 21h Service 3Ch
*/
int  Create( char _far *name, unsigned int attr ) {

    errnum = 0;

    _asm {
        push    ds
        lds     dx, name
        mov     cx, attr
        mov     ah, 3ch
        int     21h
        pop     ds
        jnc     Cr_ok
        mov     errnum, ax
        mov     ax, -1
Cr_ok:
        nop
    }
}

/*
;   Open File
;   Interrupt 21h Service 3Dh
*/
int  Open( char *name, int attr ) {

    errnum = 0;

    _asm {
        push    ds
        lds     dx, name
        mov     al, byte ptr attr
        mov     ah, 3dh
        int     21h
        pop     ds
        jnc     Op_ok
        mov     errnum, ax
        mov     ax, -1
Op_ok:
        nop
    }
}


unsigned char _Access( char *name, int attr ) {

    int fptr;

    fptr = Open( name, attr );
    if( fptr == -1 )
        return FALSE;
    Close( fptr );
    return TRUE;
}

/*
;   Close File
;   Interrupt 21h Service 3Eh
*/
int  Close( int fptr ) {

    errnum = 0;

    _asm {
        mov     bx, fptr
        mov     ah, 3eh
        int     21h
        jnc     Cl_ok
        mov     errnum, ax
        mov     ax, -1
Cl_ok:
        nop
    }
}

/*
;   Read File/Device
;   Interrupt 21h Service 3Fh
*/
unsigned int _Read( int fptr, char _far *buf, unsigned int num ) {

    errnum = 0;

    _asm {
        push    ds
        mov     bx, fptr
        mov     cx, num
        lds     dx, buf
        mov     ah, 3fh
        int     21h
        pop     ds
        jnc     Re_ok
        mov     errnum, ax
        mov     ax, -1
Re_ok:
        nop
    }
}

/*
 *  ReadLine - Reads a line of text from a ASCII-file.
 *             Returns length of line + 1, 0 for EOF
 */
unsigned int _ReadLine( int fptr, char _far *buf, unsigned int maxlen ) {

    int i=0, ret;

    buf[0] = 0;
    for(i=0; i<maxlen; ++i) {
        ret=_Read(fptr, &buf[i], 1);
        if ((ret==0) && (i==0)) {
            /* end of file AND empty line */
            break;
        }
        if (ret<0) {
            /* read error OR end of file */
            break;
        }
        if(buf[i] == 0x0d) {
            /* carriage return, skip line feed (0x0a) */
            _Read(fptr, &buf[i], 1);
            buf[i++]=0;
            break;
        }
    }
    buf[i] = 0;
    return i;
}


/*
;   Write File/Device
;   Interrupt 21h Service 40h
;
*/
unsigned int  _Write( int fptr, char _far *buf, unsigned int num ) {

    errnum = 0;

    _asm {
        push    ds
        mov     bx, fptr
        mov     cx, num
        lds     dx, buf
        mov     ah, 40h
        int     21h
        pop     ds
        jnc     Wr_ok
        mov     errnum, ax
        mov     ax, -1
Wr_ok:
        nop
    }
}


/*
;   Delete File
;   Interrupt 21h Service 41h
;
*/
int  Delete( char _far *name ) {

    errnum = 0;

    _asm {
        pusha
        push    ds
        mov     ah, 41h
        lds     dx, name
        int     21h
        pop     ds
        popa
        jnc     noerr
        mov     errnum, ax
        xor     ax, ax
noerr:
        nop
    }
}

/*
;   Set File Pointer
;   Interrupt 21h Service 42h
;
*/
unsigned long  Seek( int fptr, unsigned long offset, unsigned int method ) {

    errnum = 0;

    _asm {
        mov     bx, fptr
        mov     cx, word ptr[bp+10]
        mov     dx, word ptr[bp+8]
        mov     ax, method
        mov     ah, 42h
        int     21h
        jnc     Se_ok
        mov     errnum, ax
        mov     ax, -1
        xor     dx, dx
Se_ok:
        nop
    }
}

unsigned int  GetFileAttr( char _far *name ) {

    unsigned int attr;

    _asm {
        push    ds
        lds     dx, name
        mov     ah, 43h
        xor     al, al

        int     21h
        pop     ds
        mov     ax, cx
    }
}


/*
;   Get Current Directory
;   Interrupt 21h Service 47h
;
*/
unsigned int GetCurrDir( char _far *dir ) {

    errnum = 0;

    _asm {
        push    ds
        mov     ah, 19h     ; get current disk
        int     21h
        mov     dl, al
        add     al, 'A'
        lds     si, dir
        mov     byte ptr[si], al
        mov     word ptr[si+1], 5c3ah
        add     si, 3
        add     dl, 1
        mov     ah, 47h
        int     21h
        pop     ds
        jnc     noc
        mov     errnum, ax
        xor     ax, ax
    }

    dir[0] = 0;

    _asm {
        jmp     done
noc:
        mov     ax, 1
done:
        nop
    }
}

/*
;   Find First File
;   Interrupt 21h Service 4Eh
;
*/
unsigned int  FindFirstFile( char _far *name, unsigned int attr ) {

    errnum = 0;

    _asm {
        push    ds
        mov     cx, attr
        lds     dx, name

        mov     ah, 4eh
        int     21h
        pop     ds

        jnc     noc
        mov     errnum, ax
        xor     ax, ax
        jmp     done
noc:
        mov     ax, 1
done:
        nop
    }
}

/*
;   Find Next File
;   Interrupt 21h Service 4Fh
;
*/
unsigned int  FindNextFile( void ) {

    errnum = 0;

    _asm {
        mov     ah, 4fh
        int     21h

        jnc     nocFNF
        mov     errnum, ax
        xor     ax, ax
        jmp     doneFNF
nocFNF:
        mov     ax, 1
doneFNF:
        nop
    }
}


unsigned int Rename( char *curr, char *new ) {

    errnum = 0;

    _asm {
        push    ds
        mov     ah, 56h
        lds     dx, curr
        les     di, new
        int     21h
        pop     ds
        jc      error
        xor     ax, ax
        jmp     noerr
error:
        mov     errnum, ax
noerr:
        nop
    }
}


unsigned long GetFileDate( int fptr ) {

    _asm {
        mov     ah, 57h
        xor     al, al
        mov     bx, fptr
        int     21h

        jnc     noerr
        xor     dx, dx
        xor     cx, cx
noerr:
        mov     ax, cx
    }
}

int SetFileDate( unsigned long date, int fptr ) {

    _asm {
        mov     ah, 57h
        mov     al, 1
        mov     bx, fptr
        mov     cx, word ptr date
        mov     dx, word ptr date+2
        int     21h
    }
}


/*
; ********************************************************************
; *
; *     ERROR SERVICES
; *
; *
; ********************************************************************
*/
/*
;
;   Get Extended Error Information
;   Interrupt 21h Service 59h
;
*/
#define DMB_OK      (1<<0)
#define DMB_RETRY   (1<<3)
#define DMB_CANCEL  (1<<4)

static char internalerr[] = "Internal Error";

static struct _doserror {
    char *msg;
    unsigned int flags;
} doserr[] = {
    "No Error",                                DMB_OK,
    internalerr,                               DMB_OK,
    "File Not Found",                          DMB_OK,
    "Path Not Found",                          DMB_OK,
    "Too Mony Open Files",                     DMB_OK,
    "Access Denied",                           DMB_OK,
    "Handle Invalid",                          DMB_OK,
    "Memory Control Block Destroyed",          DMB_OK,
    "Insufficient Memory",                     DMB_OK,
    "Memory Block Address Invalid",            DMB_OK,
    "Environment Invalid",                     DMB_OK,
    "Format Invalid",                          DMB_OK,
    "Access Code Invalid",                     DMB_OK,
    "Data Invalid",                            DMB_OK,
    "Unknown Unit",                            DMB_OK,
    "Disk Drive Invalid",                      DMB_OK,
    internalerr,                               DMB_OK,
    "Not Same Device",                         DMB_OK,
    internalerr,                               DMB_OK,
    "Disk Write-Protected",                    DMB_RETRY|DMB_CANCEL,
    "Unknown Unit",                            DMB_OK,
    "Drive Not Ready",                         DMB_RETRY|DMB_CANCEL,
    "Unknown Command",                         DMB_OK,
    "Data Error (CRC)",                        DMB_RETRY|DMB_CANCEL,
    internalerr,                               DMB_OK,
    "Seek Error",                              DMB_RETRY|DMB_CANCEL,
    "Unknown Media Type",                      DMB_RETRY|DMB_CANCEL,
    "Sector Not Found",                        DMB_RETRY|DMB_CANCEL,
    "Printer Out of Paper",                    DMB_RETRY|DMB_CANCEL,
    "Write Fault",                             DMB_RETRY|DMB_CANCEL,
    "Read Fault",                              DMB_RETRY|DMB_CANCEL,
    "General Failure",                         DMB_RETRY|DMB_CANCEL,
    "Sharing Violation",                       DMB_OK,
    "Lock Violation",                          DMB_RETRY|DMB_CANCEL,
    "Disk Change Invalid",                     DMB_RETRY|DMB_CANCEL,
    internalerr,                               DMB_OK,
    "Sharing Buffer Exceeded",                 DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    "Print Queue Full",                        DMB_RETRY|DMB_CANCEL,
    "Not Enough Space for Print File",         DMB_RETRY|DMB_CANCEL,
    "Print File Canceled",                     DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    "File Sharing Temporarily Paused",         DMB_RETRY|DMB_CANCEL,
    internalerr,                               DMB_OK,
    "Print or Disk Redirection Paused",        DMB_RETRY|DMB_CANCEL,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    "File Already Exists",                     DMB_OK,
    internalerr,                               DMB_OK,
    "Cannot Make Directory",                   DMB_OK,
    "Fail On Int 24h",                         DMB_OK,
    "Too Many Redirections",                   DMB_OK,
    "Duplicate Redirection",                   DMB_OK,
    "Invalid Password",                        DMB_OK,
    "Invalid Parameter",                       DMB_OK,
    internalerr,                               DMB_OK,
    internalerr,                               DMB_OK,
    "Required System Component Not Installed", DMB_OK,
};

static unsigned int geterror( void );

static unsigned int geterror( void ) {

    if( errnum )
        return errnum;

    _asm {
        push    ds
        mov     ah, 59h
        mov     bx, 0
        int     21h
        pop     ds
    }
}

unsigned int  GetExtError( void ) {

    return geterror();
}


struct DOSError *DOSError( int err ) {

    static struct DOSError de;

    if( err == -1 )
        de.Code = geterror();
    else
        de.Code = err;

    if( de.Code >= sizeof(doserr)/sizeof(struct _doserror) )
        de.Code = 1;

    de.Msg = doserr[de.Code].msg;
    de.Flags = doserr[de.Code].flags;

    return &de;
}
